from _collections import defaultdict
from _functools import reduce
from _weakrefset import WeakSet
import collections
import itertools
import operator
import random
import time
import weakref
from animation.posture_manifest_constants import PostureConstants
from autonomy.autonomy_gsi_enums import AutonomyStageLabel, GSIDataKeys
from autonomy.autonomy_interaction_priority import AutonomyInteractionPriority
from autonomy.autonomy_mixer_provider import _MixerProvider, _MixerProviderType
from autonomy.autonomy_mixer_provider_scoring import _MixerProviderScoring
from autonomy.autonomy_modifier_enums import SuppressionCheckOption
from autonomy.autonomy_preference import AutonomyPreferenceType
from autonomy.autonomy_request import AutonomyDistanceEstimationBehavior, AutonomyPostureBehavior
from caches import cached
from clock import ClockSpeedMode
from event_testing.resolver import SingleSimResolver
from event_testing.results import TestResult
from interactions.aop import AffordanceObjectPair
from interactions.constraints import ANYWHERE
from interactions.context import InteractionContext, InteractionSource
from objects.components.inventory_type_tuning import InventoryTypeTuning
from sims4.tuning.geometric import TunableWeightedUtilityCurve, TunableCurve
from sims4.tuning.tunable import Tunable, TunableInterval, TunableSimMinute, TunableMapping, TunableEnumEntry, TunableReference, TunableRealSecond, TunedInterval
from statistics.base_statistic import StatisticChangeDirection
from tag import Tag
import autonomy.autonomy_exceptions
import autonomy.autonomy_util
import autonomy.settings
import clock
import date_and_time
import element_utils
import elements
import gsi_handlers
import interactions
import objects.components.types
import services
import sims4.log
import sims4.random
import sims4.reload
import sims4.repr_utils
import sims4.resources
import singletons
logger = sims4.log.Logger('Autonomy', default_owner='rez')
timeslice_logger = sims4.log.Logger('AutonomyTimeslice', default_owner='rez')
gsi_logger = sims4.log.Logger('AutonomyGSI', default_owner='iobaid')
ScoredInteractionData = collections.namedtuple('ScoredInteractionData', ['score', 'route_time', 'multitasking_percentage', 'interaction'])
_DeferredAopData = collections.namedtuple('_DeferredAopData', ['aop', 'inventory_type', 'outside_multiplier'])

class AutonomyMode:
    FULL_AUTONOMY_DELAY = TunableInterval(TunableSimMinute, 15, 30, minimum=0, description='\n                                  Amount of time, in sim minutes, between full autonomy runs.  System will randomize between \n                                  min and max each time')
    FULL_AUTONOMY_DELAY_WITH_NO_PRIMARY_SIS = TunableInterval(TunableSimMinute, 1.5, 2.5, minimum=0, description="\n                                                      The amount of time, in sim minutes, to wait before running autonomy if a sim \n                                                      is not in any primary SI's and hasn't run a user-directed action since \n                                                      AUTONOMY_DELAY_AFTER_USER_INTERACTION.")
    FULL_AUTONOMY_DELAY_WITH_NO_RESULT = TunableInterval(description="\n                                                The amount of time, in sim minutes, to wait before running autonomy if a sim's \n                                                autonomy returned None.\n                                                ", tunable_type=TunableSimMinute, default_lower=20, default_upper=30, minimum=1)
    AUTONOMY_DELAY_AFTER_USER_INTERACTION = TunableSimMinute(25, description='\n                                                    The amount of time, in sim minutes, before a sim that performs a user-direction \n                                                    interaction will run autonomy.')
    MAX_REAL_SECONDS_UNTIL_TIMESLICING_IS_REMOVED = TunableRealSecond(description='\n                                                        The amount of time before autonomy stops timeslicing and forces the autonomy request to \n                                                        run unimpeded.', default=1)
    FULL_AUTONOMY_STATISTIC_SCORE_VARIANCE = Tunable(float, 0.9, description='\n                                                     The percentage variance that a statistic can have from the top stat before it is \n                                                     not considered for the first round of scoring.')
    FULL_AUTONOMY_OPTIMISTIC_SCORE_THRESHOLD = Tunable(float, 0.5, description="\n                                                     Before computing the routing distance to an object, we calculate an 'optimistic' score,\n                                                     which assumes that an object is right next to a Sim and won't invalidate any of their\n                                                     current interactions.  If the optimistic score is lower than this fraction of the best\n                                                     scoring affordance so far the Sim will not bother estimating routing distance.")
    FULL_AUTONOMY_MULTIPLIER_FOR_SOLVING_THE_SAME_STAT = Tunable(float, 0.25, description='\n                                                     If a sim is currently solving a motive, this value will be multiplied into the \n                                                     commodity score of any other interactions.  This will force sims to live with \n                                                     their decisions rather than always looking for the best thing.')
    FULL_AUTONOMY_DESIRE_TO_JOIN_PLAYER_PARTIES = Tunable(float, 0, description='\n                                                          This weight is multiplied with the affordance score if the target party has \n                                                          any sims that are not autonomous.')
    FULL_AUTONOMY_ATTENTION_COST = TunableWeightedUtilityCurve(description='\n                                                             A curve that maps the total attention cost with a score multiplier.  This value will be \n                                                             multiplied with the typical autonomy score to account for multi-tasking costs.')
    FULL_AUTONOMY_MULTITASKING_PERCENTAGE_BONUS = TunableCurve(description='\n                                                             A curve that maps the commodity desire score with a percentage bonus applied to the \n                                                             base multitasking chance.')
    FULL_AUTONOMY_INTERACTION_PRIORITY_RANKING = TunableMapping(description='\n        Mapping to decide the order autonomous interaction will be chosen for\n        full autonomy ping.\n        ', key_type=TunableEnumEntry(AutonomyInteractionPriority, AutonomyInteractionPriority.INVALID, invalid_enums=(AutonomyInteractionPriority.INVALID,)), value_type=Tunable(int, 0))
    OFF_LOT_OBJECT_SCORE_MULTIPLIER = Tunable(description='\n                                                The autonomy score multiplier for off-lot object when a sim \n                                                is on the active lot.\n                                                ', tunable_type=float, default=0.5)
    SUBACTION_AUTONOMY_CONTENT_SCORE_UTILITY_CURVE = TunableWeightedUtilityCurve(description='\n                                                             A curve that maps the content score to the provided utility.')
    SUBACTION_MOTIVE_UTILITY_FALLBACK_SCORE = Tunable(float, 0.1, description='\n                                                      If score for sub-action motive utility is zero, use this value as the best score.')
    SUBACTION_GROUP_UNTUNED_WEIGHT = 1
    SUBACTION_GROUP_WEIGHTING = TunableMapping(description='\n                                    Mapping of mixer interaction group tags to scores.  This is used by subaction autonomy\n                                    to decide which set of mixers to run.  See the mixer_group entry in the sub_action tunable\n                                    on any mixer for details as to how the system works.\n                                    ', key_type=TunableEnumEntry(interactions.MixerInteractionGroup, interactions.MixerInteractionGroup.DEFAULT, description='\n                                        The mixer group this score applies to.\n                                        '), value_type=Tunable(int, SUBACTION_GROUP_UNTUNED_WEIGHT, description='\n                                        The weight of this group.\n                                        '))
    POSTURE_CHANGE_OPPORTUNITY_COST_MULTIPLIER = Tunable(float, 1.5, description='\n                                                        Multiplier to apply to the total opportunity cost of an aop when choosing that aop would\n                                                        force the Sim to change postures.  This makes the concept of changing postures less \n                                                        attractive to Sims.')
    AUTOMATED_RANDOMIZATION_LIST = TunableMapping(description='\n        A mapping of the commodities used for determinisitc randomization.  This is used by the automation\n        system in the min spec perf tests.\n        ', key_type=TunableReference(description='\n            The statistic we are operating on.', manager=services.get_instance_manager(sims4.resources.Types.STATISTIC)), value_type=Tunable(description='\n            The number of times per loop to assign this to a Sim.', tunable_type=int, default=1))
    NUMBER_OF_DUPLICATE_AFFORDANCE_TAGS_TO_SCORE = Tunable(description='\n                                                        If an affordance is tuned with duplicate_affordance_group set to anything \n                                                        except INVALID, this is the number of affordances that share this tag that\n                                                        will be scored. \n                                                        ', tunable_type=int, default=3)
    _full_autonomy_delay_override = None
    _autonomy_delay_after_user_interaction_override = None
    _disable_autonomous_multitasking_if_user_directed_override = singletons.DEFAULT
    _MINIMUM_SCORE = 1e-05

    def __init__(self, request):
        self._request = request
        self._motive_scores = None
        self._process_start_time = None

    def __str__(self):
        return 'Unknown Mode'

    @property
    def _sim(self):
        return self._request.sim

    def run_gen(self, timeline, timeslice):
        self._motive_scores = self._score_motives()
        self._request.create_autonomy_ping_request_record()
        result = yield from self._run_gen(timeline, timeslice)
        self._request.add_record_to_profiling_data()
        return result

    def _run_gen(self, timeline, timeslice):
        raise NotImplementedError
        yield None

    @classmethod
    def toggle_disable_autonomous_multitasking_if_user_directed(cls, to_enabled):
        if cls._disable_autonomous_multitasking_if_user_directed_override is singletons.DEFAULT:
            is_enabled = False
        else:
            is_enabled = cls._disable_autonomous_multitasking_if_user_directed_override
        if to_enabled is None:
            cls._disable_autonomous_multitasking_if_user_directed_override = not is_enabled
        else:
            cls._disable_autonomous_multitasking_if_user_directed_override = to_enabled
        return cls._disable_autonomous_multitasking_if_user_directed_override

    @classmethod
    def get_autonomy_delay_after_user_interaction(cls):
        return clock.interval_in_sim_minutes(cls._get_autonomy_delay_after_user_interaction_in_sim_minutes())

    @classmethod
    def _get_autonomy_delay_after_user_interaction_in_sim_minutes(cls):
        if cls._autonomy_delay_after_user_interaction_override is not None:
            return cls._autonomy_delay_after_user_interaction_override
        else:
            return cls.AUTONOMY_DELAY_AFTER_USER_INTERACTION

    @classmethod
    def get_autonomous_delay_time(cls):
        return clock.interval_in_sim_minutes(cls._get_autonomous_delay_time_in_sim_minutes())

    @classmethod
    def _get_autonomous_delay_time_in_sim_minutes(cls):
        raise NotImplementedError

    @classmethod
    def get_autonomous_update_delay_with_no_primary_sis(cls):
        return clock.interval_in_sim_minutes(cls._get_autonomous_update_delay_with_no_primary_sis_in_sim_minutes())

    @classmethod
    def _get_autonomous_update_delay_with_no_primary_sis_in_sim_minutes(cls):
        return cls.FULL_AUTONOMY_DELAY_WITH_NO_PRIMARY_SIS.random_float()

    @classmethod
    def get_no_result_delay_time(cls):
        return clock.interval_in_sim_minutes(cls.FULL_AUTONOMY_DELAY_WITH_NO_RESULT.random_float())

    @classmethod
    def override_full_autonomy_delay(cls, lower_bound, upper_bound):
        if lower_bound > upper_bound:
            logger.error('lower_bound > upper_bound in override_full_autonomy_delay()')
        else:
            cls._full_autonomy_delay_override = TunedInterval(lower_bound=lower_bound, upper_bound=upper_bound)

    @classmethod
    def clear_full_autonomy_delay_override(cls):
        cls._full_autonomy_delay_override = None

    @classmethod
    def override_full_autonomy_delay_after_user_action(cls, delay):
        cls._autonomy_delay_after_user_interaction_override = delay

    @classmethod
    def clear_full_autonomy_delay_after_user_action(cls):
        cls._autonomy_delay_after_user_interaction_override = None

    @classmethod
    def test(cls, sim):
        return True

    @classmethod
    def is_silent_mode(cls):
        return False

    def set_process_start_time(self):
        self._process_start_time = time.clock()

    @classmethod
    def allows_routing(cls):
        return False

    def _allow_autonomous(self, aop):
        if self._request.ignore_user_directed_and_autonomous:
            return TestResult.TRUE
        affordance = aop.affordance
        if self._request.context.source == InteractionContext.SOURCE_AUTONOMY:
            if not affordance.allow_autonomous:
                if self._request.record_test_result is not None:
                    self._request.record_test_result(aop, 'allow_autonomous', None)
                return TestResult(False, 'allow_autonomous is False.')
            if affordance.is_super and not affordance.commodity_flags:
                if self._request.record_test_result is not None:
                    self._request.record_test_result(aop, 'allow_autonomous', None)
                return TestResult(False, 'No commodities were advertised.')
        if self._request.context.source == InteractionContext.SOURCE_PIE_MENU and not affordance.allow_user_directed:
            if self._request.record_test_result is not None:
                self._request.record_test_result(aop, 'allow_user_directed', None)
            return TestResult(False, 'allow_user_directed is False.')
        return TestResult.TRUE

    def _is_available(self, obj):
        context = self._request.context
        if context.source != context.SOURCE_AUTONOMY and not context.always_check_in_use:
            return self._get_object_result(obj, success=True, reason='Scored')
        if self._request.ignore_lockouts or self._sim.has_lockout(obj):
            if self._request.record_test_result is not None:
                self._request.record_test_result(None, '_is_available', sims4.utils.Result(False, 'Sim has lockout.'))
            return self._get_object_result(obj, success=False, reason='Sim has lockout for this object')
        return self._get_object_result(obj, success=True, reason='Scored')

    def _get_object_result(self, obj, success:bool, reason:str):
        return success

    def _score_motives(self):
        motive_scores = None
        motive_scores = {stat.stat_type: ScoredStatistic(stat, self._sim) for stat in tuple(self._sim.scored_stats_gen())}
        if self._request.has_commodities:
            for stat in self._request.all_commodities:
                stat_type = stat.stat_type
                if stat_type not in motive_scores:
                    statistic_instance = self._sim.get_stat_instance(stat_type, add=False) or stat_type
                    motive_scores[stat_type] = ScoredStatistic(statistic_instance, self._sim)
        return motive_scores

    @classmethod
    def _should_log(cls, active_sim):
        if not cls.is_silent_mode():
            if active_sim is not None:
                return services.autonomy_service().should_log(active_sim)
            else:
                return False


class _MixerAutonomy(AutonomyMode):

    def __init__(self, request):
        super().__init__(request)
        self._mixer_provider_scoring = None
        self._mixer_provider_and_group_to_scored_mixer_aops = {}
        self._gsi_mixer_scoring = None
        self._run_gen_call_count = 0

    def _cache_mixer_provider_scoring(self, gsi_enabled_at_start):
        if self._mixer_provider_scoring is not None:
            return
        self._mixer_provider_scoring = _MixerProviderScoring(gsi_enabled_at_start=gsi_enabled_at_start)
        for mixer_provider in self._mixer_providers_gen():
            if not mixer_provider.has_mixers():
                pass
            else:
                (mixer_providing_score, gsi_mixer_provider_data) = self._score_mixer_provider(mixer_provider, gsi_enabled_at_start)
                self._mixer_provider_scoring.add_mixer_provider(mixer_providing_score, mixer_provider, gsi_mixer_provider_data)

    def _run_gen(self, timeline, timeslice):
        gsi_enabled_at_start = gsi_handlers.autonomy_handlers.archiver.enabled
        if self._gsi_mixer_scoring is None:
            self._gsi_mixer_scoring = []
        self._cache_mixer_provider_scoring(gsi_enabled_at_start)
        self._run_gen_call_count += 1
        scored_mixers_aops = None
        while gsi_enabled_at_start and self._mixer_provider_scoring.is_valid():
            chosen_mixer_provider = self._mixer_provider_scoring.get_mixer_provider()
            if chosen_mixer_provider is None:
                return
            interaction_mixer_group_weight = []
            for mixer_interaction_group in chosen_mixer_provider.mixer_interaction_groups():
                if not self._mixer_provider_scoring.is_mixer_group_valid(chosen_mixer_provider, mixer_interaction_group):
                    pass
                elif mixer_interaction_group in self.SUBACTION_GROUP_WEIGHTING:
                    interaction_mixer_group_weight.append((self.SUBACTION_GROUP_WEIGHTING[mixer_interaction_group], mixer_interaction_group))
                else:
                    logger.error('Untuned weight for mixer group: {}', mixer_interaction_group)
                    interaction_mixer_group_weight.append((self.SUBACTION_GROUP_UNTUNED_WEIGHT, mixer_interaction_group))
            mixer_interaction_group = None
            if interaction_mixer_group_weight:
                mixer_interaction_group = sims4.random.weighted_random_item(interaction_mixer_group_weight)
            else:
                self._mixer_provider_scoring.remove_invalid_mixer_provider(chosen_mixer_provider)
            if mixer_interaction_group is None:
                pass
            else:
                scored_mixers_aops = self._mixer_provider_and_group_to_scored_mixer_aops.get((chosen_mixer_provider, mixer_interaction_group), None)
                if scored_mixers_aops is not None:
                    break

                def _aop_pre_tests(aop):
                    return self._allow_autonomous(aop)

                mixer_aops = chosen_mixer_provider.get_mixers(self._request, mixer_interaction_group, aop_pre_tests=_aop_pre_tests)
                if mixer_aops is None:
                    self._mixer_provider_scoring.invalidate_group_for_mixer_provider(chosen_mixer_provider, mixer_interaction_group)
                    if self._gsi_mixer_scoring is not None:
                        self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'No Mixers Available', self._run_gen_call_count)
                        scored_mixers_aops = self._create_and_score_mixers(chosen_mixer_provider, mixer_aops, self._gsi_mixer_scoring)
                        if scored_mixers_aops is None:
                            self._mixer_provider_scoring.invalidate_group_for_mixer_provider(chosen_mixer_provider, mixer_interaction_group)
                            if self._gsi_mixer_scoring is not None:
                                self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'No Mixers Scored', self._run_gen_call_count)
                                self._mixer_provider_and_group_to_scored_mixer_aops[(chosen_mixer_provider, mixer_interaction_group)] = scored_mixers_aops
                                if self._gsi_mixer_scoring is not None:
                                    self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'Scored', self._run_gen_call_count)
                                break
                        else:
                            self._mixer_provider_and_group_to_scored_mixer_aops[(chosen_mixer_provider, mixer_interaction_group)] = scored_mixers_aops
                            if self._gsi_mixer_scoring is not None:
                                self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'Scored', self._run_gen_call_count)
                            break
                else:
                    scored_mixers_aops = self._create_and_score_mixers(chosen_mixer_provider, mixer_aops, self._gsi_mixer_scoring)
                    if scored_mixers_aops is None:
                        self._mixer_provider_scoring.invalidate_group_for_mixer_provider(chosen_mixer_provider, mixer_interaction_group)
                        if self._gsi_mixer_scoring is not None:
                            self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'No Mixers Scored', self._run_gen_call_count)
                            self._mixer_provider_and_group_to_scored_mixer_aops[(chosen_mixer_provider, mixer_interaction_group)] = scored_mixers_aops
                            if self._gsi_mixer_scoring is not None:
                                self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'Scored', self._run_gen_call_count)
                            break
                    else:
                        self._mixer_provider_and_group_to_scored_mixer_aops[(chosen_mixer_provider, mixer_interaction_group)] = scored_mixers_aops
                        if self._gsi_mixer_scoring is not None:
                            self._mixer_provider_scoring.add_mixer_provider_mixer_result_to_gsi(chosen_mixer_provider, mixer_interaction_group, 'Scored', self._run_gen_call_count)
                        break
        if scored_mixers_aops is None:
            return
        valid_interactions = ValidInteractions()
        for (interaction_score, mixer_aop) in scored_mixers_aops.values():
            if self._request.skipped_affordance_list and mixer_aop.affordance in self._request.skipped_affordance_list:
                pass
            else:
                mixer_result = mixer_aop.interaction_factory(self._request.context)
                if not mixer_result:
                    pass
                else:
                    mixer_interaction = mixer_result.interaction
                    if mixer_interaction is not None:
                        valid_interactions.add(ScoredInteractionData(interaction_score, 0, 1, mixer_interaction))
        self._request.valid_interactions = valid_interactions
        if not self._request.gsi_data:
            self._request.gsi_data = {GSIDataKeys.REQUEST_KEY: self._request.get_gsi_data(), GSIDataKeys.MIXERS_KEY: self._gsi_mixer_scoring, GSIDataKeys.MIXER_PROVIDER_KEY: self._mixer_provider_scoring.gsi_mixer_provider_data, GSIDataKeys.COMMODITIES_KEY: self._motive_scores.values(), GSIDataKeys.OBJECTS_KEY: [], GSIDataKeys.PROBABILITY_KEY: [], GSIDataKeys.AFFORDANCE_KEY: []}
        return valid_interactions

    @classmethod
    def test(cls, sim):
        autonomy_state = sim.get_autonomy_state_setting()
        if autonomy_state <= autonomy.settings.AutonomyState.DISABLED:
            return False
        return True

    @classmethod
    def _get_autonomous_delay_time_in_sim_minutes(cls):
        return cls.SUB_ACTION_AUTONOMY_DELAY

    def _mixer_providers_gen(self):
        raise NotImplementedError
        yield None

    def _score_mixer_provider(self, mixer_provider, gsi_enabled_at_start=False):
        scored_commodity = mixer_provider.get_scored_commodity(self._motive_scores)
        motive_utility = None
        if scored_commodity:
            motive_utility = self._motive_scores.get(scored_commodity)
        if motive_utility is None:
            motive_utility = self.SUBACTION_MOTIVE_UTILITY_FALLBACK_SCORE
        else:
            motive_utility = max(motive_utility, self.SUBACTION_MOTIVE_UTILITY_FALLBACK_SCORE)
        score = motive_utility*mixer_provider.get_subaction_weight()
        gsi_mixer_provider_data = None
        if gsi_enabled_at_start:
            score_detail_string = 'Weight: {}; ScoredCommodity: {} Commodity Score: {}'.format(mixer_provider.get_subaction_weight(), scored_commodity, motive_utility)
            gsi_mixer_provider_data = gsi_handlers.autonomy_handlers.GSIMixerProviderData(str(mixer_provider), mixer_provider.target_string, score, score_detail_string)
        return (score, gsi_mixer_provider_data)

    def _create_and_score_mixers(self, mixer_provider, mixer_aops, gsi_mixer_scoring):
        mixer_scores = None
        content_set_score = 1
        mixer_provider_is_social = mixer_provider.is_social
        club_service = services.get_club_service()
        business_service = services.business_service()
        business_manager = business_service.get_business_manager_for_zone(services.current_zone_id())
        for (mixer_weight, mixer_aop, _) in mixer_aops:
            mixer_aop_affordance = mixer_aop.affordance
            mixer_aop_target = mixer_aop.target
            if mixer_provider_is_social and (mixer_aop_affordance.target_type & interactions.TargetType.TARGET and mixer_aop_target.is_sim) and mixer_aop_target.disallow_as_mixer_target:
                pass
            else:
                if mixer_provider_is_social:
                    content_set_score = self._get_subaction_content_set_utility_score(mixer_aop.content_score)
                    score = mixer_weight*content_set_score
                    if club_service is not None:
                        club_score_multiplier = club_service.get_interaction_score_multiplier(mixer_aop, sim=self._sim)
                        score *= club_score_multiplier
                    if business_manager is not None:
                        business_score_multiplier = business_manager.get_interaction_score_multiplier(mixer_aop, sim=self._sim)
                        score *= business_score_multiplier
                else:
                    score = mixer_weight
                if score <= self._MINIMUM_SCORE:
                    score = 0
                if mixer_scores is None:
                    mixer_scores = {}
                mixer_scores[mixer_aop_affordance] = (score, mixer_aop)
                if gsi_mixer_scoring is not None:
                    score_detail_string = 'Weight: {}; Content Score: {}; Group: {}'.format(mixer_weight, content_set_score, mixer_aop_affordance.sub_action.mixer_group)
                    gsi_mixer_scoring.append((score, str(mixer_provider), str(mixer_aop_affordance), str(mixer_aop_target), score_detail_string))
        return mixer_scores

    def _get_subaction_content_set_utility_score(self, content_score):
        if content_score is None:
            return self.SUBACTION_MOTIVE_UTILITY_FALLBACK_SCORE
        else:
            utility = self.SUBACTION_AUTONOMY_CONTENT_SCORE_UTILITY_CURVE.get(content_score)
            if utility == 0:
                return self.SUBACTION_MOTIVE_UTILITY_FALLBACK_SCORE
        return utility


class SubActionAutonomy(_MixerAutonomy):

    def __str__(self):
        return 'SubActionAutonomy'

    def _mixer_providers_gen(self):
        super_affordance_mixer_override = False
        for si in self._sim.si_state:
            if not si.only_use_mixers_from_si:
                pass
            else:
                super_affordance_mixer_override = True
                if not si.has_affordances():
                    pass
                else:
                    mixer_provider = _MixerProvider(si, _MixerProviderType.SI)
                    yield mixer_provider
        if not super_affordance_mixer_override:
            for si in self._sim.si_state:
                if not si.has_affordances():
                    pass
                else:
                    mixer_provider = _MixerProvider(si, _MixerProviderType.SI)
                    yield mixer_provider
            for buff in self._sim.Buffs:
                if buff.interactions is None:
                    pass
                else:
                    mixer_provider = _MixerProvider(buff, _MixerProviderType.BUFF)
                    yield mixer_provider


class SocialAutonomy(_MixerAutonomy):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

    def __str__(self):
        return 'SocialAutonomy'

    def _run_gen(self, timeline, timeslice):
        if not self._validate_request():
            return
        result = yield from super()._run_gen(timeline, timeslice)
        return result

    def _validate_request(self):
        if not self._request.static_commodity_list:
            logger.error('Failed to run SocialAutonomy: no static commodities listed.')
            return False
        elif not self._request.object_list:
            logger.error('Failed to run SocialAutonomy: no objects listed.')
            return False
        return True

    def _mixer_providers_gen(self):
        for target_sim in self._request.objects_to_score_gen(self._request.static_commodity_list):
            if not target_sim.is_sim:
                pass
            elif not self._is_available(target_sim):
                pass
            else:
                for affordance in target_sim.super_affordances():
                    for static_commodity in self._request.static_commodity_list:
                        if static_commodity in affordance.static_commodities:
                            break
                    aop = AffordanceObjectPair(affordance, target_sim, affordance, None)
                    execute_result = aop.interaction_factory(self._request.context)
                    if not execute_result:
                        logger.error('Failed to create interaction: '.format(aop))
                    else:
                        interaction = execute_result.interaction
                        incompatible_sis = self._sim.si_state.get_incompatible(interaction)
                        if incompatible_sis:
                            interaction.invalidate()
                        else:
                            (target_interaction, test_result) = interaction.get_target_si()
                            if not test_result:
                                interaction.invalidate()
                            else:
                                if target_interaction is None:
                                    target_context = self._request.context.clone_for_sim(target_sim)
                                    target_sim_si_state = target_sim.si_state
                                    incompatible_sis.add(interaction)
                                else:
                                    incompatible_sis = target_sim.si_state.get_incompatible(target_interaction)
                                if incompatible_sis:
                                    interaction.invalidate()
                                else:
                                    self._request.interactions_to_invalidate.append(interaction)
                                    yield _MixerProvider(interaction, _MixerProviderType.SI)


class _SuperInteractionAutonomy(AutonomyMode):
    UNREACHABLE_DESTINATION_COST = 1000000

    def __init__(self, request):
        super().__init__(request)
        self._actively_scored_motives = None
        self._formerly_scored_motives = set()

    def _test_aop(self, aop):
        context = self._request.context
        test_result = aop.test(context)
        if not test_result:
            if self._request.record_test_result is not None:
                self._request.record_test_result(aop, '_test_aop', test_result)
            return test_result
        return TestResult.TRUE

    def _has_available_part(self, interaction):
        if interaction.target is None:
            return True
        if not interaction.target.parts:
            return True
        for part in interaction.target.parts:
            reservation_handler = interaction.get_interaction_reservation_handler(target=part)
            if reservation_handler is None:
                return True
            if reservation_handler.may_reserve():
                return True
        return False

    def _calculate_route_time_and_opportunity(self, timeline, interaction):
        request = self._request
        if interaction.disable_distance_estimation_and_posture_checks:
            return (0, 0, False, set())
        ignore_included_sis = request.posture_behavior == AutonomyPostureBehavior.IGNORE_SI_STATE
        if request.distance_estimation_behavior == AutonomyDistanceEstimationBehavior.FINAL_PATH:
            (estimated_distance, must_change_posture, included_sis) = yield from interaction.estimate_final_path_distance(timeline, ignore_included_sis)
        elif ignore_included_sis:
            (estimated_distance, must_change_posture, included_sis) = interaction.estimate_distance_ignoring_other_sis()
        else:
            (estimated_distance, must_change_posture, included_sis) = interaction.estimate_distance()
        if request.distance_estimation_behavior == AutonomyDistanceEstimationBehavior.ALLOW_UNREACHABLE_LOCATIONS and estimated_distance is None:
            estimated_distance = _SuperInteractionAutonomy.UNREACHABLE_DESTINATION_COST
        elif estimated_distance is not None:
            estimated_distance = 0
        route_time = None
        if estimated_distance is not None:
            route_time = estimated_distance*date_and_time.REAL_MILLISECONDS_PER_SIM_SECOND/date_and_time.TICKS_PER_REAL_WORLD_SECOND
        return (route_time, estimated_distance, must_change_posture, included_sis)

    def _satisfies_active_desire(self, aop):
        commodity_flags = aop.affordance.commodity_flags
        if commodity_flags & self._formerly_scored_motives:
            return False
        if not self._actively_scored_motives:
            return True
        elif commodity_flags & self._actively_scored_motives:
            return True
        return False

    @staticmethod
    def _is_valid_interaction(interaction):
        if interaction.affordance.autonomy_can_overwrite_similar_affordance or interaction.sim.si_state.is_affordance_active_for_actor(interaction.affordance):
            return TestResult(False, 'Sim is already running the same affordance.')
        return TestResult.TRUE

    def _get_object_result(self, obj, success:bool, reason:str):
        if success:
            return ObjectResult.Success(obj, relevant_desires=self._actively_scored_motives)
        else:
            return ObjectResult.Failure(obj, relevant_desires=self._actively_scored_motives, reason=reason)

    @classmethod
    def allows_routing(cls):
        return True


class ValidInteractions:

    def __init__(self):
        self._data = {}
        self._refs = {}
        self._counts = {}

    def __str__(self):
        return str(self._data)

    def __repr__(self):
        return sims4.repr_utils.standard_angle_repr(self, self._data)

    def __bool__(self):
        if self._data:
            return True
        return False

    def __contains__(self, affordance):
        return affordance.get_affordance_key_for_autonomy() in self._data

    def __getitem__(self, affordance):
        return self._data[affordance.get_affordance_key_for_autonomy()]

    def add(self, scored_interaction_data):
        interaction = scored_interaction_data.interaction
        affordance_key = interaction.affordance.get_affordance_key_for_autonomy()
        self._counts[affordance_key] = 1
        return self._add_internal(interaction, affordance_key, scored_interaction_data)

    def maybe_add(self, scored_interaction_data):
        interaction = scored_interaction_data.interaction
        affordance_key = interaction.affordance.get_affordance_key_for_autonomy()
        previous_data = self._data.get(affordance_key, None)
        if previous_data is None or previous_data.score < scored_interaction_data.score:
            self._counts[affordance_key] = 1
            return self._add_internal(interaction, affordance_key, scored_interaction_data)
        if previous_data.score > scored_interaction_data.score:
            return interaction
        else:
            c = self._counts[affordance_key]
            c += 1
            self._counts[affordance_key] = c
            if random.random() < 1/c:
                return self._add_internal(interaction, affordance_key, scored_interaction_data)
            else:
                return interaction
        return interaction

    def _add_internal(self, interaction, affordance_key, scored_interaction_data):
        previous_data = self._data.get(affordance_key)
        self._data[affordance_key] = scored_interaction_data
        if interaction.target is not None:

            def callback(_, self_ref=weakref.ref(self)):
                self = self_ref()
                if self is not None:
                    del self._data[affordance_key]
                    del self._refs[affordance_key]

            self._refs[affordance_key] = interaction.target.ref(callback)
        else:
            self._refs[affordance_key] = None
        if previous_data is not None:
            return previous_data.interaction

    def has_score_for_aop(self, aop):
        affordance_key = aop.affordance.get_affordance_key_for_autonomy()
        if affordance_key not in self._data:
            return False
        scored_interaction_data = self._data[affordance_key]
        return scored_interaction_data.interaction.aop.is_equivalent_to(aop)

    def get_result_scores(self):
        return tuple(self._data.values())


def _dont_timeslice_gen(timeline):
    return False


class FullAutonomy(_SuperInteractionAutonomy):
    _GSI_IGNORES_NON_AUTONOMOUS_AFFORDANCES = True

    def __init__(self, request):
        super().__init__(request)
        self._relationship_object_value = 0
        self._found_valid_interaction = False
        self._inventory_posture_score_cache = {}
        self._motives_being_solved = None
        self._found_motives = set()
        self._scored_shared_inventory_types = set()
        self._valid_interactions = defaultdict(ValidInteractions)
        self._limited_affordances = defaultdict(list)
        self._gsi_objects = None
        self._gsi_interactions = None
        self._gsi_commodities_pass = 1
        self._timestamp_when_timeslicing_was_removed = None

    def _clean_up(self):
        for i in AutonomyInteractionPriority:
            if i not in self._valid_interactions:
                pass
            else:
                for scored_interaction_data in self._valid_interactions[i].get_result_scores():
                    if not scored_interaction_data.interaction.is_super:
                        pass
                    else:
                        scored_interaction_data.interaction.invalidate()

    def __str__(self):
        return 'FullAutonomy'

    def _run_gen(self, timeline, timeslice):
        if self._should_log(self._sim):
            logger.debug('Processing {}', self._sim)
        gsi_enabled_at_start = gsi_handlers.autonomy_handlers.archiver.enabled
        if gsi_enabled_at_start:
            if self._gsi_objects is None:
                self._gsi_objects = []
            if self._gsi_interactions is None:
                self._gsi_interactions = []
        (self._actively_scored_motives, motives_to_score) = self._get_motives_to_score()
        if self._actively_scored_motives or not self._request.object_list:
            return
        self._motives_being_solved = self._get_all_motives_currently_being_solved()
        if timeslice is None:
            timeslice_if_needed_gen = _dont_timeslice_gen
        else:
            start_time = time.clock()

            def timeslice_if_needed_gen(timeline):
                nonlocal start_time
                time_now = time.clock()
                elapsed_time = time_now - start_time
                if elapsed_time < timeslice:
                    return False
                if self._timestamp_when_timeslicing_was_removed is not None:
                    enable_long_slice = False
                else:
                    total_elapsed_time = time_now - self._process_start_time
                    if total_elapsed_time > self.MAX_REAL_SECONDS_UNTIL_TIMESLICING_IS_REMOVED:
                        timeslice_logger.debug('Autonomy request for {} took too long; timeslicing is removed.', self._sim)
                        self._timestamp_when_timeslicing_was_removed = time_now
                        enable_long_slice = False
                    else:
                        enable_long_slice = True
                start_time_now = time.time()
                if enable_long_slice:
                    sleep_element = element_utils.sleep_until_next_tick_element()
                else:
                    sleep_element = elements.SleepElement(date_and_time.TimeSpan(0))
                yield timeline.run_child(sleep_element)
                self._request.on_end_of_time_slicing(start_time_now)
                if self._sim is None or not self._request.valid:
                    self._clean_up()
                    raise autonomy.autonomy_exceptions.AutonomyExitException()
                start_time = time.clock()
                return True

        best_threshold = None
        while True:
            self._inventory_posture_score_cache = {}
            self._scored_shared_inventory_types.clear()
            objects_to_score = WeakSet(self._request.objects_to_score_gen(self._actively_scored_motives))
            while True:
                yield from timeslice_if_needed_gen(timeline)
                try:
                    obj = objects_to_score.pop()
                except KeyError:
                    break
                (object_result, best_threshold) = yield from self._score_object_interactions_gen(timeline, obj, timeslice_if_needed_gen, None, best_threshold)
                if self._gsi_objects is not None:
                    self._gsi_objects.append(object_result.get_log_data())
                if not obj.is_sim:
                    inventory_component = obj.inventory_component
                    if self._should_score_object_inventory(inventory_component):
                        best_threshold = yield from self._score_object_inventory_gen(timeline, inventory_component, timeslice_if_needed_gen, best_threshold)
            for aop_list in self._limited_affordances.values():
                valid_aop_list = [aop_data for aop_data in aop_list if aop_data.aop.target is not None]
                num_aops = len(valid_aop_list)
                if num_aops > self.NUMBER_OF_DUPLICATE_AFFORDANCE_TAGS_TO_SCORE:
                    final_aop_list = []
                    for aop_data in valid_aop_list:
                        autonomy_preference = aop_data.aop.affordance.autonomy_preference
                        if autonomy_preference is not None and self._sim.get_autonomy_preference_type(autonomy_preference.preference.tag, aop_data.aop.target, True, allow_test=False) >= AutonomyPreferenceType.USE_PREFERENCE:
                            final_aop_list.append(aop_data)
                    num_final_aops = len(final_aop_list)
                    if num_final_aops < self.NUMBER_OF_DUPLICATE_AFFORDANCE_TAGS_TO_SCORE:
                        non_preferred_aops = [aop_data for aop_data in valid_aop_list if aop_data not in final_aop_list]
                        if non_preferred_aops:
                            final_aop_list.extend(random.sample(non_preferred_aops, self.NUMBER_OF_DUPLICATE_AFFORDANCE_TAGS_TO_SCORE - num_final_aops))
                else:
                    final_aop_list = valid_aop_list
                if self._gsi_interactions is not None:
                    for aop_data in valid_aop_list:
                        if aop_data not in final_aop_list:
                            self._gsi_interactions.append(InteractionResult.Failure(aop_data.aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Randomly discarded due to limited autonomy for {}'.format(aop_data.aop.affordance.duplicate_affordance_group)))
                for aop_data in final_aop_list:
                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop_data.aop, aop_data.inventory_type, best_threshold, aop_data.outside_multiplier)
                    if not interaction_result:
                        if self._request.record_test_result is not None:
                            self._request.record_test_result(aop_data.aop, '_create_and_score_interaction', interaction_result)
                        if self._gsi_interactions is not None:
                            self._gsi_interactions.append(interaction_result)
                            (_, best_threshold) = self._process_scored_interaction(aop_data.aop, interaction, interaction_result, route_time, best_threshold)
                    else:
                        (_, best_threshold) = self._process_scored_interaction(aop_data.aop, interaction, interaction_result, route_time, best_threshold)
            self._limited_affordances.clear()
            if not motives_to_score:
                break
            if self._actively_scored_motives:
                for stat_type in self._actively_scored_motives:
                    scored_stat = self._motive_scores[stat_type]
                    if scored_stat.gsi_commodities_pass == ScoredStatistic.UNSET_COMMODITY_PASS:
                        scored_stat.gsi_commodities_pass = self._gsi_commodities_pass
                self._gsi_commodities_pass += 1
            self._formerly_scored_motives.update(self._actively_scored_motives)
            variance_score = self._motive_scores[motives_to_score[0]]
            for motive in self._found_motives:
                variance_score = max(variance_score, self._motive_scores[motive])
            variance_score *= AutonomyMode.FULL_AUTONOMY_STATISTIC_SCORE_VARIANCE
            self._actively_scored_motives = {stat.stat_type for stat in itertools.takewhile(lambda desire: self._motive_scores[desire] >= variance_score, motives_to_score)}
            if not (self._gsi_objects is not None and self._actively_scored_motives):
                break
            if self._found_valid_interaction:
                motives_to_score = []
            else:
                motives_to_score = motives_to_score[len(self._actively_scored_motives):]
        final_valid_interactions = None
        for i in AutonomyInteractionPriority:
            if i not in self._valid_interactions:
                pass
            else:
                valid_interactions = self._valid_interactions[i]
                if valid_interactions:
                    final_valid_interactions = valid_interactions
                    break
        self._request.valid_interactions = final_valid_interactions
        if self._gsi_interactions is not None:
            self._request.gsi_data = {GSIDataKeys.REQUEST_KEY: self._request.get_gsi_data(), GSIDataKeys.MIXERS_KEY: [], GSIDataKeys.MIXER_PROVIDER_KEY: None, GSIDataKeys.OBJECTS_KEY: self._gsi_objects, GSIDataKeys.PROBABILITY_KEY: [], GSIDataKeys.AFFORDANCE_KEY: self._gsi_interactions, GSIDataKeys.COMMODITIES_KEY: self._motive_scores.values()}
        return final_valid_interactions

    @classmethod
    def _get_autonomous_delay_time_in_sim_minutes(cls):
        if cls._full_autonomy_delay_override is None:
            return cls.FULL_AUTONOMY_DELAY.random_float()
        else:
            return random.uniform(cls._full_autonomy_delay_override.lower_bound, cls._full_autonomy_delay_override.upper_bound)

    @classmethod
    def test(cls, sim):
        if services.game_clock_service().clock_speed == ClockSpeedMode.SUPER_SPEED3:
            return TestResult(False, 'In or has super speed three request')
        if sim.get_autonomy_state_setting() <= autonomy.settings.AutonomyState.LIMITED_ONLY:
            return TestResult(False, 'Limited autonomy and below can never run full autonomy')
        if sim.get_autonomy_state_setting() == autonomy.settings.AutonomyState.MEDIUM and sim.si_state.has_user_directed_si():
            return TestResult(False, 'Medium Autonomy but has a user directed interaction in si state.')
        if sim.queue is None:
            logger.warn('sim.queue is None in FullAutonomy.test()', owner='rez')
            return TestResult(False, 'Sim Partially destroyed.')
        result = cls._test_pending_interactions(sim)
        if not result:
            return result
        if sim.is_player_active():
            return TestResult(False, 'Sim actively being played.')
        master = sim.routing_master
        if master is not None and master.is_sim:
            slave_data = master.get_formation_data_for_slave(sim)
            master_path = master.routing_component.current_path
            if master_path is not None and slave_data.should_slave_for_path(master_path):
                return TestResult(False, 'Slaved to {}, who is routing.'.format(master))
            if master.transition_controller is not None:
                transition_spec = master.transition_controller.get_transition_spec(master)
                if transition_spec is not None and transition_spec.path is not None and slave_data.should_slave_for_path(transition_spec.path):
                    return TestResult(False, 'Slaved to {}, who wants to route somewhere.'.format(master))
        return TestResult.TRUE

    def _is_available(self, obj):
        super_result = super()._is_available(obj)
        if not super_result:
            return super_result
        if self._request.radius_to_consider_squared > 0:
            delta = obj.intended_position - self._sim.intended_position
            if delta.magnitude_squared() > self._request.radius_to_consider_squared:
                return self._get_object_result(obj, success=False, reason='Target object is too far away from the sim.')
        context = self._request.context
        if context.source == context.SOURCE_AUTONOMY:
            for interaction in tuple(obj.si_state):
                if interaction.disallows_full_autonomy(self._disable_autonomous_multitasking_if_user_directed_override):
                    return self._get_object_result(obj, success=False, reason='Target sim is running an interaction that disallows multi tasking.')
        return super_result

    @cached
    def _get_outside_score_modifications(self, obj, sim):
        if self._request.context.source != InteractionSource.AUTONOMY:
            return (False, 1.0)
        if obj.is_in_sim_inventory(sim):
            return (False, 1.0)
        if obj.is_outside:
            return sim.sim_info.get_outside_object_score_modification()
        elif obj.is_sim and sim.is_outside:
            return obj.sim_info.get_outside_object_score_modification()
        return (False, 1.0)

    def _should_score_object_inventory(self, inventory_component):
        if inventory_component is None:
            return False
        if not inventory_component.should_score_contained_objects_for_autonomy:
            return False
        else:
            inventory_type = inventory_component.inventory_type
            if InventoryTypeTuning.is_shared_between_objects(inventory_type) and inventory_type in self._scored_shared_inventory_types:
                return False
        return True

    def _score_object_inventory_gen(self, timeline, inventory_component, timeslice_if_needed_gen, best_threshold):
        inventory_type = inventory_component.inventory_type
        for inventory_obj in inventory_component.get_items_for_autonomy_gen(motives=self._actively_scored_motives):
            (object_result, best_threshold) = yield from self._score_object_interactions_gen(timeline, inventory_obj, timeslice_if_needed_gen, inventory_type, best_threshold)
            if self._gsi_objects is not None:
                self._gsi_objects.append(object_result.get_log_data())
        if InventoryTypeTuning.is_shared_between_objects(inventory_type):
            self._scored_shared_inventory_types.add(inventory_type)
        return best_threshold

    def _get_potential_interactions(self, obj):
        return tuple(obj.potential_interactions(self._request.context, **self._request.kwargs))

    def _score_object_interactions_gen(self, timeline, obj, timeslice_if_needed_gen, inventory_type, best_threshold):
        context = self._request.context
        obj_ref = obj.ref()
        is_available = self._is_available(obj)
        if not is_available:
            return (is_available, best_threshold)
        potential_interactions = self._get_potential_interactions(obj)
        if not potential_interactions:
            return (ObjectResult.Success(obj, relevant_desires=self._actively_scored_motives), best_threshold)
        (is_outside_supressed, object_outside_multiplier) = self._get_outside_score_modifications(obj, self._sim)
        autonomy_constraints = None
        for aop in potential_interactions:
            yielded_due_to_timeslice = yield from timeslice_if_needed_gen(timeline)
            if yielded_due_to_timeslice and obj_ref() is None:
                return (ObjectResult.Failure(obj, self._actively_scored_motives, 'Object deleted.'), best_threshold)
            if not aop.affordance.is_super:
                if context.sim is not self._sim:
                    logger.error('A non-super interaction was returned from potential_interactions(): {}', aop, owner='rez')
                    outside_multiplier = object_outside_multiplier
                    if not aop.affordance.counts_as_inside:
                        if is_outside_supressed:
                            if self._gsi_interactions is not None:
                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Sim has an outside lock which prevents it from running interaction'))
                                if self._request.affordance_list is not None and aop.affordance not in self._request.affordance_list:
                                    pass
                                elif aop.affordance.target_type == interactions.TargetType.ACTOR and self._sim is not obj:
                                    pass
                                elif not self._satisfies_active_desire(aop):
                                    if self._request.record_test_result is not None:
                                        self._request.record_test_result(aop, '_satisfies_desire', None)
                                    if self._gsi_interactions is not None:
                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Failed to satisfy relevant desires: {}', reason_args=(aop.affordance.commodity_flags,)))
                                        test_result = self._allow_autonomous(aop)
                                        if not test_result:
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, '_allow_autonomous', None)
                                            if self._GSI_IGNORES_NON_AUTONOMOUS_AFFORDANCES or self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason="aop doesn't advertise to autonomy."))
                                                if self._valid_interactions[aop.affordance.scoring_priority].has_score_for_aop(aop):
                                                    pass
                                                elif self._request.skipped_affordance_list and aop.affordance in self._request.skipped_affordance_list:
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'skipped_affordance_list', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Affordance in skipped_affordance_list'))
                                                        if self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_satisfies_desire', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                                self._relationship_object_value = 0
                                                                if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                                    crafter_id = obj.get_crafting_process().crafter_sim_id
                                                                    if crafter_id is not None:
                                                                        relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                                        if relationship_track:
                                                                            relationship_score = relationship_track.get_value()
                                                                            logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                            self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                                test_result = self._test_aop(aop)
                                                                if not test_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_test_aop', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                                        if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                                if autonomy_constraints is None:
                                                                                    autonomy_constraints = ANYWHERE
                                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                            if not autonomy_constraints.valid:
                                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                                break
                                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                                if autonomy_constraints is not ANYWHERE:
                                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                                    if aop_constraint is not None:
                                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                        if not aop_constraint.valid:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                                else:
                                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                                    if not interaction_result:
                                                                                                        if self._request.record_test_result is not None:
                                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                        if self._gsi_interactions is not None:
                                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                                    else:
                                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                            else:
                                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                                if not interaction_result:
                                                                                                    if self._request.record_test_result is not None:
                                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                    if self._gsi_interactions is not None:
                                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                                else:
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            if autonomy_constraints is None:
                                                                                autonomy_constraints = ANYWHERE
                                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                        if not autonomy_constraints.valid:
                                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                            break
                                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                            if autonomy_constraints is not ANYWHERE:
                                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                                if aop_constraint is not None:
                                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                    if not aop_constraint.valid:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                            else:
                                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                                if not interaction_result:
                                                                                                    if self._request.record_test_result is not None:
                                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                    if self._gsi_interactions is not None:
                                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                                else:
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                        if autonomy_constraints is None:
                                                                            autonomy_constraints = ANYWHERE
                                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                    if not autonomy_constraints.valid:
                                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                        break
                                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                        if autonomy_constraints is not ANYWHERE:
                                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                            if aop_constraint is not None:
                                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                if not aop_constraint.valid:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            self._relationship_object_value = 0
                                                            if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                                crafter_id = obj.get_crafting_process().crafter_sim_id
                                                                if crafter_id is not None:
                                                                    relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                                    if relationship_track:
                                                                        relationship_score = relationship_track.get_value()
                                                                        logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                        self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                            test_result = self._test_aop(aop)
                                                            if not test_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_test_aop', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                                    if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                            if autonomy_constraints is None:
                                                                                autonomy_constraints = ANYWHERE
                                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                        if not autonomy_constraints.valid:
                                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                            break
                                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                            if autonomy_constraints is not ANYWHERE:
                                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                                if aop_constraint is not None:
                                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                    if not aop_constraint.valid:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                            else:
                                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                                if not interaction_result:
                                                                                                    if self._request.record_test_result is not None:
                                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                    if self._gsi_interactions is not None:
                                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                                else:
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        if autonomy_constraints is None:
                                                                            autonomy_constraints = ANYWHERE
                                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                    if not autonomy_constraints.valid:
                                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                        break
                                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                        if autonomy_constraints is not ANYWHERE:
                                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                            if aop_constraint is not None:
                                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                if not aop_constraint.valid:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                elif self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, '_satisfies_desire', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                        self._relationship_object_value = 0
                                                        if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                            crafter_id = obj.get_crafting_process().crafter_sim_id
                                                            if crafter_id is not None:
                                                                relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                                if relationship_track:
                                                                    relationship_score = relationship_track.get_value()
                                                                    logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                    self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                        test_result = self._test_aop(aop)
                                                        if not test_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_test_aop', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                                if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                        if autonomy_constraints is None:
                                                                            autonomy_constraints = ANYWHERE
                                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                    if not autonomy_constraints.valid:
                                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                        break
                                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                        if autonomy_constraints is not ANYWHERE:
                                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                            if aop_constraint is not None:
                                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                if not aop_constraint.valid:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    self._relationship_object_value = 0
                                                    if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                        crafter_id = obj.get_crafting_process().crafter_sim_id
                                                        if crafter_id is not None:
                                                            relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                            if relationship_track:
                                                                relationship_score = relationship_track.get_value()
                                                                logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                    test_result = self._test_aop(aop)
                                                    if not test_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_test_aop', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                            if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        elif self._valid_interactions[aop.affordance.scoring_priority].has_score_for_aop(aop):
                                            pass
                                        elif self._request.skipped_affordance_list and aop.affordance in self._request.skipped_affordance_list:
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, 'skipped_affordance_list', None)
                                            if self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Affordance in skipped_affordance_list'))
                                                if self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, '_satisfies_desire', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                        self._relationship_object_value = 0
                                                        if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                            crafter_id = obj.get_crafting_process().crafter_sim_id
                                                            if crafter_id is not None:
                                                                relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                                if relationship_track:
                                                                    relationship_score = relationship_track.get_value()
                                                                    logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                    self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                        test_result = self._test_aop(aop)
                                                        if not test_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_test_aop', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                                if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                        if autonomy_constraints is None:
                                                                            autonomy_constraints = ANYWHERE
                                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                    if not autonomy_constraints.valid:
                                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                        break
                                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                        if autonomy_constraints is not ANYWHERE:
                                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                            if aop_constraint is not None:
                                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                if not aop_constraint.valid:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    self._relationship_object_value = 0
                                                    if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                        crafter_id = obj.get_crafting_process().crafter_sim_id
                                                        if crafter_id is not None:
                                                            relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                            if relationship_track:
                                                                relationship_score = relationship_track.get_value()
                                                                logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                    test_result = self._test_aop(aop)
                                                    if not test_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_test_aop', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                            if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        elif self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, '_satisfies_desire', None)
                                            if self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                self._relationship_object_value = 0
                                                if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                    crafter_id = obj.get_crafting_process().crafter_sim_id
                                                    if crafter_id is not None:
                                                        relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                        if relationship_track:
                                                            relationship_score = relationship_track.get_value()
                                                            logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                            self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                test_result = self._test_aop(aop)
                                                if not test_result:
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, '_test_aop', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                        if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        else:
                                            self._relationship_object_value = 0
                                            if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                crafter_id = obj.get_crafting_process().crafter_sim_id
                                                if crafter_id is not None:
                                                    relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                    if relationship_track:
                                                        relationship_score = relationship_track.get_value()
                                                        logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                        self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                            test_result = self._test_aop(aop)
                                            if not test_result:
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, '_test_aop', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                    if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                if autonomy_constraints is None:
                                                    autonomy_constraints = ANYWHERE
                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                            if not autonomy_constraints.valid:
                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                break
                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                if autonomy_constraints is not ANYWHERE:
                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                    if aop_constraint is not None:
                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                        if not aop_constraint.valid:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                else:
                                    test_result = self._allow_autonomous(aop)
                                    if not test_result:
                                        if self._request.record_test_result is not None:
                                            self._request.record_test_result(aop, '_allow_autonomous', None)
                                        if self._GSI_IGNORES_NON_AUTONOMOUS_AFFORDANCES or self._gsi_interactions is not None:
                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason="aop doesn't advertise to autonomy."))
                                            if self._valid_interactions[aop.affordance.scoring_priority].has_score_for_aop(aop):
                                                pass
                                            elif self._request.skipped_affordance_list and aop.affordance in self._request.skipped_affordance_list:
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, 'skipped_affordance_list', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Affordance in skipped_affordance_list'))
                                                    if self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_satisfies_desire', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                            self._relationship_object_value = 0
                                                            if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                                crafter_id = obj.get_crafting_process().crafter_sim_id
                                                                if crafter_id is not None:
                                                                    relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                                    if relationship_track:
                                                                        relationship_score = relationship_track.get_value()
                                                                        logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                        self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                            test_result = self._test_aop(aop)
                                                            if not test_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_test_aop', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                                    if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                            if autonomy_constraints is None:
                                                                                autonomy_constraints = ANYWHERE
                                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                        if not autonomy_constraints.valid:
                                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                            break
                                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                            if autonomy_constraints is not ANYWHERE:
                                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                                if aop_constraint is not None:
                                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                    if not aop_constraint.valid:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                            else:
                                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                                if not interaction_result:
                                                                                                    if self._request.record_test_result is not None:
                                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                    if self._gsi_interactions is not None:
                                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                                else:
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        if autonomy_constraints is None:
                                                                            autonomy_constraints = ANYWHERE
                                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                    if not autonomy_constraints.valid:
                                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                        break
                                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                        if autonomy_constraints is not ANYWHERE:
                                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                            if aop_constraint is not None:
                                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                if not aop_constraint.valid:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        self._relationship_object_value = 0
                                                        if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                            crafter_id = obj.get_crafting_process().crafter_sim_id
                                                            if crafter_id is not None:
                                                                relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                                if relationship_track:
                                                                    relationship_score = relationship_track.get_value()
                                                                    logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                    self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                        test_result = self._test_aop(aop)
                                                        if not test_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_test_aop', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                                if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                        if autonomy_constraints is None:
                                                                            autonomy_constraints = ANYWHERE
                                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                    if not autonomy_constraints.valid:
                                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                        break
                                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                        if autonomy_constraints is not ANYWHERE:
                                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                            if aop_constraint is not None:
                                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                                if not aop_constraint.valid:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                        else:
                                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                            if not interaction_result:
                                                                                                if self._request.record_test_result is not None:
                                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                                if self._gsi_interactions is not None:
                                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                            else:
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            elif self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, '_satisfies_desire', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                    self._relationship_object_value = 0
                                                    if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                        crafter_id = obj.get_crafting_process().crafter_sim_id
                                                        if crafter_id is not None:
                                                            relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                            if relationship_track:
                                                                relationship_score = relationship_track.get_value()
                                                                logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                    test_result = self._test_aop(aop)
                                                    if not test_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_test_aop', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                            if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                self._relationship_object_value = 0
                                                if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                    crafter_id = obj.get_crafting_process().crafter_sim_id
                                                    if crafter_id is not None:
                                                        relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                        if relationship_track:
                                                            relationship_score = relationship_track.get_value()
                                                            logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                            self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                test_result = self._test_aop(aop)
                                                if not test_result:
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, '_test_aop', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                        if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                    elif self._valid_interactions[aop.affordance.scoring_priority].has_score_for_aop(aop):
                                        pass
                                    elif self._request.skipped_affordance_list and aop.affordance in self._request.skipped_affordance_list:
                                        if self._request.record_test_result is not None:
                                            self._request.record_test_result(aop, 'skipped_affordance_list', None)
                                        if self._gsi_interactions is not None:
                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Affordance in skipped_affordance_list'))
                                            if self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, '_satisfies_desire', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                    self._relationship_object_value = 0
                                                    if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                        crafter_id = obj.get_crafting_process().crafter_sim_id
                                                        if crafter_id is not None:
                                                            relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                            if relationship_track:
                                                                relationship_score = relationship_track.get_value()
                                                                logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                    test_result = self._test_aop(aop)
                                                    if not test_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_test_aop', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                            if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                self._relationship_object_value = 0
                                                if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                    crafter_id = obj.get_crafting_process().crafter_sim_id
                                                    if crafter_id is not None:
                                                        relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                        if relationship_track:
                                                            relationship_score = relationship_track.get_value()
                                                            logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                            self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                test_result = self._test_aop(aop)
                                                if not test_result:
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, '_test_aop', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                        if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                    elif self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                        if self._request.record_test_result is not None:
                                            self._request.record_test_result(aop, '_satisfies_desire', None)
                                        if self._gsi_interactions is not None:
                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                            self._relationship_object_value = 0
                                            if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                crafter_id = obj.get_crafting_process().crafter_sim_id
                                                if crafter_id is not None:
                                                    relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                    if relationship_track:
                                                        relationship_score = relationship_track.get_value()
                                                        logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                        self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                            test_result = self._test_aop(aop)
                                            if not test_result:
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, '_test_aop', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                    if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                if autonomy_constraints is None:
                                                    autonomy_constraints = ANYWHERE
                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                            if not autonomy_constraints.valid:
                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                break
                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                if autonomy_constraints is not ANYWHERE:
                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                    if aop_constraint is not None:
                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                        if not aop_constraint.valid:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                    else:
                                        self._relationship_object_value = 0
                                        if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                            crafter_id = obj.get_crafting_process().crafter_sim_id
                                            if crafter_id is not None:
                                                relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                if relationship_track:
                                                    relationship_score = relationship_track.get_value()
                                                    logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                    self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                        test_result = self._test_aop(aop)
                                        if not test_result:
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, '_test_aop', None)
                                            if self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                            if self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                if autonomy_constraints is None:
                                                    autonomy_constraints = ANYWHERE
                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                            if not autonomy_constraints.valid:
                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                break
                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                if autonomy_constraints is not ANYWHERE:
                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                    if aop_constraint is not None:
                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                        if not aop_constraint.valid:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        else:
                                            if autonomy_constraints is None:
                                                autonomy_constraints = ANYWHERE
                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                        if not autonomy_constraints.valid:
                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                            break
                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                            if autonomy_constraints is not ANYWHERE:
                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                if aop_constraint is not None:
                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                    if not aop_constraint.valid:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                if duplicate_affordance_group != Tag.INVALID:
                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                else:
                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                    if not interaction_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(interaction_result)
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                    else:
                        outside_multiplier = 1.0
                    if (is_outside_supressed or outside_multiplier != 1.0) and self._request.affordance_list is not None and aop.affordance not in self._request.affordance_list:
                        pass
                    elif aop.affordance.target_type == interactions.TargetType.ACTOR and self._sim is not obj:
                        pass
                    elif not self._satisfies_active_desire(aop):
                        if self._request.record_test_result is not None:
                            self._request.record_test_result(aop, '_satisfies_desire', None)
                        if self._gsi_interactions is not None:
                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Failed to satisfy relevant desires: {}', reason_args=(aop.affordance.commodity_flags,)))
                            test_result = self._allow_autonomous(aop)
                            if not test_result:
                                if self._request.record_test_result is not None:
                                    self._request.record_test_result(aop, '_allow_autonomous', None)
                                if self._GSI_IGNORES_NON_AUTONOMOUS_AFFORDANCES or self._gsi_interactions is not None:
                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason="aop doesn't advertise to autonomy."))
                                    if self._valid_interactions[aop.affordance.scoring_priority].has_score_for_aop(aop):
                                        pass
                                    elif self._request.skipped_affordance_list and aop.affordance in self._request.skipped_affordance_list:
                                        if self._request.record_test_result is not None:
                                            self._request.record_test_result(aop, 'skipped_affordance_list', None)
                                        if self._gsi_interactions is not None:
                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='Affordance in skipped_affordance_list'))
                                            if self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, '_satisfies_desire', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                                    self._relationship_object_value = 0
                                                    if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                        crafter_id = obj.get_crafting_process().crafter_sim_id
                                                        if crafter_id is not None:
                                                            relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                            if relationship_track:
                                                                relationship_score = relationship_track.get_value()
                                                                logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                                self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                    test_result = self._test_aop(aop)
                                                    if not test_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_test_aop', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                            if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                    if autonomy_constraints is None:
                                                                        autonomy_constraints = ANYWHERE
                                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                                if not autonomy_constraints.valid:
                                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                    break
                                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                    if autonomy_constraints is not ANYWHERE:
                                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                        if aop_constraint is not None:
                                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                            if not aop_constraint.valid:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                    else:
                                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                        if not interaction_result:
                                                                                            if self._request.record_test_result is not None:
                                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                            if self._gsi_interactions is not None:
                                                                                                self._gsi_interactions.append(interaction_result)
                                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                        else:
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                self._relationship_object_value = 0
                                                if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                    crafter_id = obj.get_crafting_process().crafter_sim_id
                                                    if crafter_id is not None:
                                                        relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                        if relationship_track:
                                                            relationship_score = relationship_track.get_value()
                                                            logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                            self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                                test_result = self._test_aop(aop)
                                                if not test_result:
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, '_test_aop', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                        if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                                if autonomy_constraints is None:
                                                                    autonomy_constraints = ANYWHERE
                                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                            if not autonomy_constraints.valid:
                                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                                break
                                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                                if autonomy_constraints is not ANYWHERE:
                                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                    if aop_constraint is not None:
                                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                        if not aop_constraint.valid:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                                else:
                                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                    if not interaction_result:
                                                                                        if self._request.record_test_result is not None:
                                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                        if self._gsi_interactions is not None:
                                                                                            self._gsi_interactions.append(interaction_result)
                                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                    else:
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                    elif self._request.skipped_static_commodities and self._satisfies_desire(self._request.skipped_static_commodities, aop):
                                        if self._request.record_test_result is not None:
                                            self._request.record_test_result(aop, '_satisfies_desire', None)
                                        if self._gsi_interactions is not None:
                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.BEFORE_TESTS, self._actively_scored_motives, reason='AOP satisfies explicitly skipped commodity'))
                                            self._relationship_object_value = 0
                                            if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                                crafter_id = obj.get_crafting_process().crafter_sim_id
                                                if crafter_id is not None:
                                                    relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                    if relationship_track:
                                                        relationship_score = relationship_track.get_value()
                                                        logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                        self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                            test_result = self._test_aop(aop)
                                            if not test_result:
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, '_test_aop', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                    if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                            if autonomy_constraints is None:
                                                                autonomy_constraints = ANYWHERE
                                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                        if not autonomy_constraints.valid:
                                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                            break
                                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                            if autonomy_constraints is not ANYWHERE:
                                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                                if aop_constraint is not None:
                                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                    if not aop_constraint.valid:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                            else:
                                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                                if not interaction_result:
                                                                                    if self._request.record_test_result is not None:
                                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                    if self._gsi_interactions is not None:
                                                                                        self._gsi_interactions.append(interaction_result)
                                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                                else:
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                if self._request.record_test_result is not None:
                                                    self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                if self._gsi_interactions is not None:
                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                if autonomy_constraints is None:
                                                    autonomy_constraints = ANYWHERE
                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                            if not autonomy_constraints.valid:
                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                break
                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                if autonomy_constraints is not ANYWHERE:
                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                    if aop_constraint is not None:
                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                        if not aop_constraint.valid:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                    else:
                                        self._relationship_object_value = 0
                                        if obj.has_component(objects.components.types.CRAFTING_COMPONENT):
                                            crafter_id = obj.get_crafting_process().crafter_sim_id
                                            if crafter_id is not None:
                                                relationship_track = self._sim.sim_info.relationship_tracker.get_relationship_track(crafter_id)
                                                if relationship_track:
                                                    relationship_score = relationship_track.get_value()
                                                    logger.assert_log(relationship_track.relationship_obj_prefence_curve is not None, 'Error: Tuning for RelationshipTrack: {}, Relationship Object Preference Curve is not tuned.'.format(type(relationship_track).__name__))
                                                    self._relationship_object_value = relationship_track.relationship_obj_prefence_curve.get(relationship_score)
                                        test_result = self._test_aop(aop)
                                        if not test_result:
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, '_test_aop', None)
                                            if self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason=test_result.reason))
                                                if self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                                    if self._request.record_test_result is not None:
                                                        self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                                    if self._gsi_interactions is not None:
                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                        if autonomy_constraints is None:
                                                            autonomy_constraints = ANYWHERE
                                                            if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                                for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                    constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                    constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                    autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                    if not autonomy_constraints.valid:
                                                                        logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                        break
                                                            if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                                autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                        if autonomy_constraints is not ANYWHERE:
                                                            aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                            if aop_constraint is not None:
                                                                aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                                if not aop_constraint.valid:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                        if duplicate_affordance_group != Tag.INVALID:
                                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                        else:
                                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                            if not interaction_result:
                                                                                if self._request.record_test_result is not None:
                                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                                if self._gsi_interactions is not None:
                                                                                    self._gsi_interactions.append(interaction_result)
                                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                            else:
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    if autonomy_constraints is None:
                                                        autonomy_constraints = ANYWHERE
                                                        if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                            for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                                constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                                constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                                autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                                if not autonomy_constraints.valid:
                                                                    logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                    break
                                                        if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                            autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                    if autonomy_constraints is not ANYWHERE:
                                                        aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                        if aop_constraint is not None:
                                                            aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                            if not aop_constraint.valid:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, 'invalid_constraint', None)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                    if duplicate_affordance_group != Tag.INVALID:
                                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                    else:
                                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                        if not interaction_result:
                                                                            if self._request.record_test_result is not None:
                                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                            if self._gsi_interactions is not None:
                                                                                self._gsi_interactions.append(interaction_result)
                                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                        else:
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        elif self._request.ignore_suppressed_affordances_autonomy or obj.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.PROVIDED_AFFORDANCE_ONLY) or self._sim.sim_info.check_affordance_for_suppression(self._sim, aop, False, check_option=SuppressionCheckOption.AFFORDANCE_ONLY):
                                            if self._request.record_test_result is not None:
                                                self._request.record_test_result(aop, 'check_affordance_for_suppression', None)
                                            if self._gsi_interactions is not None:
                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_TESTS, self._actively_scored_motives, reason='aop is being supressed by something'))
                                                if autonomy_constraints is None:
                                                    autonomy_constraints = ANYWHERE
                                                    if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                        for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                            constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                            constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                            autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                            if not autonomy_constraints.valid:
                                                                logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                                break
                                                    if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                        autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                                if autonomy_constraints is not ANYWHERE:
                                                    aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                    if aop_constraint is not None:
                                                        aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                        if not aop_constraint.valid:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, 'invalid_constraint', None)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                                if duplicate_affordance_group != Tag.INVALID:
                                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                                else:
                                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                    if not interaction_result:
                                                                        if self._request.record_test_result is not None:
                                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                        if self._gsi_interactions is not None:
                                                                            self._gsi_interactions.append(interaction_result)
                                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                    else:
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                        else:
                                            if autonomy_constraints is None:
                                                autonomy_constraints = ANYWHERE
                                                if self._request.context.source != InteractionContext.SOURCE_PIE_MENU:
                                                    for tuned_constraint_entry in self._request.tuned_constraint_entries:
                                                        constraint_target = self._sim if tuned_constraint_entry.target_to_self else obj
                                                        constraint = tuned_constraint_entry.constraint.create_constraint(self._sim, constraint_target, from_autonomy=True)
                                                        autonomy_constraints = autonomy_constraints.intersect(constraint)
                                                        if not autonomy_constraints.valid:
                                                            logger.warn("Constraint provided by the sim's autonomy modifiers is invalid. Sim: {}", self._sim)
                                                            break
                                                if self._request.tuned_constraint_entries and self._request.constraint is not None:
                                                    autonomy_constraints = autonomy_constraints.intersect(self._request.constraint)
                                            if autonomy_constraints is not ANYWHERE:
                                                aop_constraint = aop.constraint_intersection(self._sim, posture_state=None)
                                                if aop_constraint is not None:
                                                    aop_constraint = aop_constraint.intersect(autonomy_constraints)
                                                    if not aop_constraint.valid:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, 'invalid_constraint', None)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(InteractionResult.Failure(aop, AutonomyStageLabel.AFTER_POSTURE_SEARCH, self._actively_scored_motives, reason='Failed constraint intersection'))
                                                            duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                            if duplicate_affordance_group != Tag.INVALID:
                                                                self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                            else:
                                                                (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                                if not interaction_result:
                                                                    if self._request.record_test_result is not None:
                                                                        self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                    if self._gsi_interactions is not None:
                                                                        self._gsi_interactions.append(interaction_result)
                                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                                else:
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                        if duplicate_affordance_group != Tag.INVALID:
                                                            self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                        else:
                                                            (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                            if not interaction_result:
                                                                if self._request.record_test_result is not None:
                                                                    self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                                if self._gsi_interactions is not None:
                                                                    self._gsi_interactions.append(interaction_result)
                                                                    (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                            else:
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                else:
                                                    duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                    if duplicate_affordance_group != Tag.INVALID:
                                                        self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                    else:
                                                        (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                        if not interaction_result:
                                                            if self._request.record_test_result is not None:
                                                                self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                            if self._gsi_interactions is not None:
                                                                self._gsi_interactions.append(interaction_result)
                                                                (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                        else:
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                            else:
                                                duplicate_affordance_group = aop.affordance.duplicate_affordance_group
                                                if duplicate_affordance_group != Tag.INVALID:
                                                    self._limited_affordances[duplicate_affordance_group].append(_DeferredAopData(aop, inventory_type, outside_multiplier))
                                                else:
                                                    (interaction_result, interaction, route_time) = yield from self._create_and_score_interaction(timeline, aop, inventory_type, best_threshold, outside_multiplier)
                                                    if not interaction_result:
                                                        if self._request.record_test_result is not None:
                                                            self._request.record_test_result(aop, '_create_and_score_interaction', interaction_result)
                                                        if self._gsi_interactions is not None:
                                                            self._gsi_interactions.append(interaction_result)
                                                            (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_time, best_threshold)
                                                    else:
                                                        (_, best_threshold) = self._process_scored_interaction(aop, interaction, interaction_result, route_